/**************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <EGL/egl.h>
#include "wayland-client.h"
#include "wayland-egl.h"
#include "ilm_clientUtil.h"
#include "WaylandServerinfoClientProtocol.h"

/****************************************************************************/
typedef struct t_wlContextStruct
{
    struct wl_display*    wlDisplay;
    struct wl_registry*   wlRegistry;
    struct wl_compositor* wlCompositor;
    struct wl_surface*    wlSurface;
    struct wl_egl_window* wlNativeWindow;
    struct serverinfo*    wlExtServerinfo;
    int      width;
    int      height;
    uint32_t connect_id;
    uint32_t surface_id;
} WLContextStruct;

static EglContextStruct g_eglContextStruct;
static WLContextStruct g_wlContextStruct;

/****************************************************************************/
static const struct wl_interface *types[] = {
        NULL,
};

static const struct wl_message serverinfo_requests[] = {
        { "get_connection_id", "", types + 0 },
};

static const struct wl_message serverinfo_events[] = {
        { "connection_id", "u", types + 0 },
};

#define ARRAY_LENGTH(a) (sizeof (a) / sizeof (a)[0])

const struct wl_interface serverinfo_interface = {
        "serverinfo", 1,
        ARRAY_LENGTH(serverinfo_requests), serverinfo_requests,
        ARRAY_LENGTH(serverinfo_events), serverinfo_events,
};

/****************************************************************************/
static void
serverinfoListener(void *data, struct serverinfo *pServerinfo, uint32_t client_handle)
{
    pServerinfo = pServerinfo; // TODO:to avoid warning
    WLContextStruct* p_wlCtx = (WLContextStruct*)data;
    p_wlCtx->connect_id = client_handle;
    fprintf( stderr, "notified wayland connection : id=%d\n", p_wlCtx->connect_id);
}

struct serverinfo_listener serverinfo_listener_list = {
    serverinfoListener
};

static void registry_handle_global(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
    version = version; // TODO: to avoid warning
    WLContextStruct* p_wlCtx = (WLContextStruct*)data;
    int ans_strcmp = 0;

    do
    {
        ans_strcmp = strcmp(interface, "wl_compositor");
        if (0 == ans_strcmp)
        {
            p_wlCtx->wlCompositor = (struct wl_compositor*)wl_registry_bind(registry, name, &wl_compositor_interface, 1);
            break;
        }

        ans_strcmp = strcmp(interface, "serverinfo");
        if (0 == ans_strcmp)
        {
            p_wlCtx->wlExtServerinfo = (struct serverinfo*)wl_registry_bind(registry, name, &serverinfo_interface, 1);
            serverinfo_add_listener(p_wlCtx->wlExtServerinfo, &serverinfo_listener_list, data);
            serverinfo_get_connection_id(p_wlCtx->wlExtServerinfo);
        }
    } while(0);
}

static const struct wl_registry_listener registry_listener = {
    registry_handle_global
};

static void frame_listener_func(void *data, struct wl_callback *callback, uint32_t time)
{
    data = data; // TODO:to avoid warning
    time = time; // TODO:to avoid warining
    if (callback)
    {
        wl_callback_destroy(callback);
    }
}

static const struct wl_callback_listener frame_listener = {
    frame_listener_func    
};

static void destroyNativeResource()
{
    if (g_wlContextStruct.wlSurface)
    {
        wl_surface_destroy(g_wlContextStruct.wlSurface);
    }

    g_wlContextStruct.wlSurface    = NULL;
}

static ilmErrorTypes createNativeResource(t_ilm_uint width, t_ilm_uint height)
{
	struct wl_proxy* pxy = NULL;

    g_wlContextStruct.width  = width;
    g_wlContextStruct.height = height;

    if (g_wlContextStruct.wlSurface == NULL)
    {
        g_wlContextStruct.wlSurface = wl_compositor_create_surface(g_wlContextStruct.wlCompositor);
        if (NULL == g_wlContextStruct.wlSurface)
        {
            ILM_ERROR("createNativeResouce", "wl_compositor_create_surface() failed.\n");
            destroyNativeResource();
            return ILM_FAILED;
        }

        pxy = (struct wl_proxy*)g_wlContextStruct.wlSurface;
        if (NULL != pxy)
        {
            g_wlContextStruct.surface_id = (uint32_t)wl_proxy_get_id(pxy);
        }
    }

    return ILM_SUCCESS;
}

/****************************************************************************/

ilmErrorTypes ilm_clientUtil_init()
{
    /* g_wlContextStruct initialize */
    g_wlContextStruct.wlDisplay       = NULL;
    g_wlContextStruct.wlRegistry      = NULL;
    g_wlContextStruct.wlCompositor    = NULL;
    g_wlContextStruct.wlSurface       = NULL;
    g_wlContextStruct.wlNativeWindow  = NULL;
    g_wlContextStruct.width           = 0;
    g_wlContextStruct.height          = 0;
    g_wlContextStruct.wlExtServerinfo = NULL;

    /* g_eglContextStruct initialize */
    g_eglContextStruct.eglDisplay = EGL_DEFAULT_DISPLAY;
    g_eglContextStruct.eglConfig  = NULL;
    g_eglContextStruct.eglSurface = EGL_NO_SURFACE;
    g_eglContextStruct.eglContext = EGL_NO_CONTEXT;

    g_wlContextStruct.wlDisplay = wl_display_connect(NULL);
    if (NULL == g_wlContextStruct.wlDisplay)
    {
        ILM_ERROR("ilm_clientUtil_init", "wl_display_connect() failed.\n");
        return ILM_FAILED;
    }

    g_wlContextStruct.wlRegistry = wl_display_get_registry(g_wlContextStruct.wlDisplay);
    if (NULL == g_wlContextStruct.wlRegistry)
    {
        ILM_ERROR("ilm_clientUtil_init", "wl_display_get_registry() failed.\n");
        wl_display_disconnect(g_wlContextStruct.wlDisplay);
        return ILM_FAILED;
    }

    wl_registry_add_listener(g_wlContextStruct.wlRegistry, &registry_listener, &g_wlContextStruct);
    wl_display_dispatch(g_wlContextStruct.wlDisplay);
    wl_display_roundtrip(g_wlContextStruct.wlDisplay);

    return ILM_SUCCESS;
}

void ilm_clientUtil_destroy()
{
    destroyNativeResource();

    if (g_wlContextStruct.wlCompositor)
    {
        wl_compositor_destroy(g_wlContextStruct.wlCompositor);
    }
    g_wlContextStruct.wlCompositor = NULL;

    if (g_wlContextStruct.wlDisplay)
    {
        wl_display_flush(g_wlContextStruct.wlDisplay);
        wl_display_disconnect(g_wlContextStruct.wlDisplay);
    }
    g_wlContextStruct.wlDisplay = NULL;
}

ilmErrorTypes ilm_createNativeWindow(t_ilm_uint width, t_ilm_uint height)
{
    if (ILM_SUCCESS != createNativeResource(width, height))
    {
        return ILM_FAILED;
    }

    g_wlContextStruct.wlNativeWindow = wl_egl_window_create(g_wlContextStruct.wlSurface, width, height);
    if (NULL == g_wlContextStruct.wlNativeWindow)
    {
        ILM_ERROR("ilm_createNativeWindow", "wl_egl_window_create() failed.\n");
        ilm_destroyNativeWindow();
        return ILM_FAILED;
    }

    return ILM_SUCCESS;
}

ilmErrorTypes ilm_createNativePixmap(t_ilm_uint width, t_ilm_uint height)
{
    ILM_ERROR("ilm_createNativePixmap", "There is no wl_egl_pixmap_create() API in wayland 1.0.0.\n");
    ILM_ERROR("ilm_createNativePixmap", "So nativeWindow is created instead of nativePixmap\n");
    return ilm_createNativeWindow(width, height);
}

ilmErrorTypes ilm_createEGLContext()
{
    g_eglContextStruct.eglDisplay = NULL;
    g_eglContextStruct.eglSurface = NULL;
    g_eglContextStruct.eglContext = NULL;
    EGLint eglstatus = EGL_SUCCESS;
    int iConfigs;
    EGLint pi32ConfigAttribs[] = {
        EGL_SURFACE_TYPE, EGL_WINDOW_BIT | EGL_PIXMAP_BIT,
        EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
        EGL_RED_SIZE,   8,
        EGL_GREEN_SIZE, 8,
        EGL_BLUE_SIZE,  8,
        EGL_ALPHA_SIZE, 8,
        EGL_NONE
    };
    EGLint contextAttribs[] = {
        EGL_CONTEXT_CLIENT_VERSION, 2,
        EGL_NONE
    };
    t_ilm_uint width  = g_wlContextStruct.width;
    t_ilm_uint height = g_wlContextStruct.height;

    g_eglContextStruct.eglDisplay = eglGetDisplay(
        (EGLNativeDisplayType)g_wlContextStruct.wlDisplay);
    eglstatus = eglGetError();
    if (!g_eglContextStruct.eglDisplay)
    {
        ILM_ERROR("ilm_createEGLContext", "eglGetDisplay() failed.\n");
    }

    EGLint iMajorVersion, iMinorVersion;
    if (!eglInitialize(g_eglContextStruct.eglDisplay, &iMajorVersion,
            &iMinorVersion))
    {
        ILM_ERROR("ilm_createEGLContext", "eglInitialize() failed.\n");
    }

    eglBindAPI(EGL_OPENGL_ES_API);
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        ILM_ERROR("ilm_createEGLContext", "eglBindAPI() failed.\n");
    }

    if (!eglChooseConfig(g_eglContextStruct.eglDisplay, pi32ConfigAttribs,
        &g_eglContextStruct.eglConfig, 1, &iConfigs) || (iConfigs != 1))
    {
        ILM_ERROR("ilm_createEGLContext", "eglChooseConfig() failed.\n");
    }
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        ILM_ERROR("ilm_createEGLContext", "eglChooseConfig() failed.\n");
    }

    if (g_wlContextStruct.wlNativeWindow != NULL)
    {
        g_eglContextStruct.eglSurface = eglCreateWindowSurface(
                g_eglContextStruct.eglDisplay, g_eglContextStruct.eglConfig,
                (EGLNativeWindowType)g_wlContextStruct.wlNativeWindow, NULL);
    }
    else
    {
        ILM_ERROR("ilm_createEGLContext",
                  "No support native pixmap in wayland 1.0.0.\n");
    }
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        ILM_ERROR("ilm_createEGLContext", "eglCreateWindowSurface() failed.\n");
    }

    g_eglContextStruct.eglContext = eglCreateContext(
            g_eglContextStruct.eglDisplay, g_eglContextStruct.eglConfig, NULL,
            contextAttribs);

    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        ILM_ERROR("ilm_createEGLContext", "eglCreateContext() failed.\n");
    }

    eglMakeCurrent(g_eglContextStruct.eglDisplay,
            g_eglContextStruct.eglSurface, g_eglContextStruct.eglSurface,
            g_eglContextStruct.eglContext);
    eglSwapInterval(g_eglContextStruct.eglDisplay, 1);
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        ILM_ERROR("ilm_createEGLContext", "eglMakeCurrent() failed.\n");
    }

    return ILM_SUCCESS;
}

void ilm_destroyNativeWindow()
{
    if (g_wlContextStruct.wlNativeWindow)
    {
        wl_egl_window_destroy(g_wlContextStruct.wlNativeWindow);
    }
    g_wlContextStruct.wlNativeWindow = NULL;
#if 0
    destroyNativeResource();
#endif
}

void ilm_destroyNativePixmap()
{
    ilm_destroyNativeWindow();
}

void ilm_destroyEGLContext()
{
    if (g_eglContextStruct.eglDisplay != NULL)
    {
        eglMakeCurrent(g_eglContextStruct.eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
        eglTerminate(g_eglContextStruct.eglDisplay);
    }
}

void ilm_swapBuffers()
{
    if (g_wlContextStruct.wlNativeWindow != NULL)
    {
        eglSwapBuffers(g_eglContextStruct.eglDisplay, g_eglContextStruct.eglSurface);
    }
    else
    {
        eglWaitClient();

        wl_surface_damage(g_wlContextStruct.wlSurface,
                          0, 0,
                          g_wlContextStruct.width,
                          g_wlContextStruct.height);
    }

    struct wl_callback* callback = wl_surface_frame(g_wlContextStruct.wlSurface);
    wl_callback_add_listener(callback, &frame_listener, NULL);
    wl_display_roundtrip(g_wlContextStruct.wlDisplay);
}

void ilm_eglSwapBuffers(EGLDisplay display, EGLSurface surface)
{
    eglSwapBuffers(display, surface);
}

NativePixmapType ilm_getNativePixmap()
{
    return (NativePixmapType)ilm_getNativeWindow();
}

NativeWindowType ilm_getNativeWindow()
{
    return (NativeWindowType)(g_wlContextStruct.wlNativeWindow);
}

NativeDisplayType ilm_getNativeDisplay()
{
    return (NativeDisplayType)(g_wlContextStruct.wlDisplay);
}

t_ilm_nativehandle ilm_getNativeHandle()
{
    return (t_ilm_nativehandle)((g_wlContextStruct.connect_id << 16) | g_wlContextStruct.surface_id);
}
